/** @file   StarGate.cpp
 * @brief   Implementation of StarGate class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/07/27 20:31:13 $
 * @author  Tomi Lamminsaari
 */

#include "StarGate.h"
#include "Settings.h"
#include "ObjectId.h"
#include "AnimId.h"
#include "GameAnims.h"
#include "eng2d.h"
#include "RedrawQueue.h"
#include "WarGlobals.h"
#include "soundsamples.h"
#include "AnimPlayer.h"
#include "SpawnItems.h"
#include "SpawnParameters.h"
#include "ObjectFactory.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

const int StarGate::KDyingCounterIndex;
const int StarGate::KSpawnCounterIndex;


///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
StarGate::StarGate() :
  GameObject(),
  iSpawnDelay(-1)
{
  int oid = ObjectID::TYPE_STARGATE;
  this->setArmor( Settings::floatObjProp( oid, "armor:") );
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  
  const Animation& anim = GameAnims::findAnimation( AnimId::KStarGate,
                                                    GameAnims::EIdle );
  this->setAnimation( anim, GameAnims::EIdle );
}



/** Destructor.
 */
StarGate::~StarGate()
{
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

void StarGate::update()
{
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( this->getCounter(KDyingCounterIndex) < 0 ) {
      this->state(GameObject::STATE_KILLED);
    }
  }
  if ( this->state() == GameObject::STATE_LIVING ) {
    if ( this->getCounter(KSpawnCounterIndex) == 0 ) {
      int posX = this->position().intX();
      int posY = this->position().intY();
      int blockX = posX / 32;
      int blockY = posY / 32;
      int offsetX = posX % 32;
      int offsetY = posY % 32;
      
      char buff[10];
      SpawnItem* item = new GameObjectSpawnItem("<alien>");
      item->iParameters.addParameter( KParamNameOffsetX, itoa(offsetX,buff,10) );
      item->iParameters.addParameter( KParamNameOffsetX, itoa(offsetY,buff,10) );
      item->iParameters.addParameter( KParamNamePosX, itoa(blockX,buff,10) );
      item->iParameters.addParameter( KParamNamePosY, itoa(blockY,buff,10) );
      item->iParameters.addParameter( KParamNameSubCategory, randomizeAlienType() );
      item->iParameters.addParameter( KParamNameIdCode, "-1" );
      item->Launch();
      delete item;
      this->setCounter( KSpawnCounterIndex, iSpawnDelay );
      
      const Animation lightAnim = GameAnims::findAnimation( AnimId::KGrenadeExplosionLight );
      AnimPlayer::spawn(lightAnim, this->position(), 0);
    }
  }
}

void StarGate::redraw( RedrawQueue* aQueue )
{
  BITMAP* frameGfx = m_animation.currentFrame().asBitmap();
  if ( frameGfx != 0 && this->hidden() == false ) {
    Vec2D pos( this->position() );
    pos -= Vec2D( Map::scrollX, Map::scrollY );
    pos -= Vec2D( frameGfx->w/2, frameGfx->h/2 );
    aQueue->addRotatedSprite(RedrawQueue::PRI_NORMAL,
                             pos.intX(), pos.intY(), itofix(this->angle()),
                             frameGfx );
  }
}

bool StarGate::hitByBullet( Bullet* aBullet )
{
  if ( aBullet->iOwner != 0 ) {
    if ( aBullet->iOwner->objectType() == ObjectID::TYPE_PLAYER ) {
      // Shooter is player.
      ParticleSparks* sparks = new ParticleSparks(aBullet->iPosition,
                                                  aBullet->velocity(), 12);
      WarGlobals::pPartManager->addSystem( sparks );
      Bullet::TType bt = aBullet->iType;
      if ( (bt == Bullet::ECrowbar || bt == Bullet::ESniperRifle) == false ) {
        aBullet->iDamage /= 4;
        this->causeDamage( aBullet );
      }
      this->makeSound( GameObject::SND_PAIN );
      return true;
    }
  }
  return false;
}

void StarGate::makeSound( GameObject::SoundID aSoundId ) const
{
  switch ( aSoundId ) {
    case ( GameObject::SND_PAIN ): {
      Sound::playSample( SMP_METALHIT, false );
      break;
    }
    case ( GameObject::SND_DIE ): {
      Sound::playSample( SMP_GRENADE, false );
      break;
    }
    default: {
      break;
    }
  }
}

void StarGate::kill()
{
  if ( this->state() != GameObject::STATE_LIVING ) {
    return;
  }
  
  this->state( GameObject::STATE_DYING );
  this->hidden( true );
  this->setCounter(KDyingCounterIndex, 120);
  
  // Create the explosion.
  const Animation& explosionAnim = GameAnims::findAnimation( AnimId::KExplosionGrenade );
  const Animation lightAnim = GameAnims::findAnimation( AnimId::KGrenadeExplosionLight );
  AnimPlayer::spawn( explosionAnim, this->position(), 0 );
  AnimPlayer::spawn( lightAnim, this->position(), 0 );
  AnimPlayer::spawn( explosionAnim, this->position() + Vec2D( 32,-32), 0 );
  AnimPlayer::spawn( lightAnim, this->position() + Vec2D( 32,-32), 0 );
  AnimPlayer::spawn( explosionAnim, this->position() + Vec2D( 32, 32), 0 );
  AnimPlayer::spawn( lightAnim, this->position() + Vec2D( 32, 32), 0 );
  AnimPlayer::spawn( explosionAnim, this->position() + Vec2D(-32, 32), 0 );
  AnimPlayer::spawn( lightAnim, this->position() + Vec2D(-32, 32), 0 );
  AnimPlayer::spawn( explosionAnim, this->position() + Vec2D(-32,-32), 0 );
  AnimPlayer::spawn( lightAnim, this->position() + Vec2D(-32,-32), 0 );
  
  // And as the sentrygun explodes, it emits some bullets
  float ac = 256.0 / 16.0;
  for (float a=0; a < 255; a+=ac) {
    eng2d::Vec2D spdVec( 0,-7 );
    spdVec.rotate( a );  
    Bullet* b = BulletTable::createBullet( this, m_position, Bullet::EGrenade );
    b->setVelocity( spdVec );
    b->iRange = 110;
    b->iDamage = 10;
    WarGlobals::pBulletManager->spawnBullet( b );
  }
  this->makeSound( GameObject::SND_DIE );
}

ObjectID::Type StarGate::objectType() const
{
  return ObjectID::TYPE_STARGATE;
}

bool StarGate::reloading() const
{
  return false;
}


///
/// New public methods
/// ============================================================================

void StarGate::setSpawnDelay( int aDelay )
{
  iSpawnDelay = aDelay;
  this->setCounter(KSpawnCounterIndex, aDelay);
}



///
/// Getter methods
/// ============================================================================




///
/// Protected and private methods
/// ============================================================================

std::string StarGate::randomizeAlienType() const
{
  int typecode = rand() % 8;
  std::string alientype = "";
  switch ( rand() % 8 ) {
    case (0):
      alientype = "winged";
      break;
    case (1):
      alientype = "proctor";
      break;
    case (2):
      alientype = "minigun";
      break;
    case (3):
      alientype = "predator";
      break;
    default:
      alientype = "carnivore";
      break;
  }
  return alientype;
}


};  // end of namespace
